<?php
/**
 * Real-Time PHP File Manager with AJAX Search and Auto-Deletion for ".htaccess"
 *
 * This script recursively scans a base directory to list all files.
 * It includes a search field that updates the file list in real time via AJAX.
 * The default search term is ".htaccess". When this default is in use (and not via AJAX),
 * the script automatically deletes all matching files without any confirmation.
 *
 * SECURITY NOTICE:
 * - Protect this script with proper authentication if deployed.
 * - Validate and sanitize all inputs in production.
 * - Automatic deletion is very dangerous; use extreme caution.
 */

$baseDir = __DIR__; // Base directory to scan. Change as needed.

/**
 * Recursively scan a directory and return an array of file paths.
 *
 * @param string $dir The directory to scan.
 * @return array List of file paths.
 */
function scanDirectoryRecursive($dir) {
    $filesArray = [];
    try {
        $directoryIterator = new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS);
        $iterator = new RecursiveIteratorIterator($directoryIterator, RecursiveIteratorIterator::SELF_FIRST);
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $filesArray[] = $file->getPathname();
            }
        }
    } catch (Exception $e) {
        error_log("Error scanning directory: " . $e->getMessage());
    }
    return $filesArray;
}

/**
 * Generate HTML table rows for files matching the search query.
 *
 * @param array $filesList List of file paths.
 * @param string $searchQuery Search query string.
 * @return string HTML string of table rows.
 */
function generateFileRows($filesList, $searchQuery) {
    // Filter files based on search query (case-insensitive)
    if ($searchQuery !== '') {
        $filesList = array_filter($filesList, function($file) use ($searchQuery) {
            return stripos($file, $searchQuery) !== false;
        });
    }
    
    $html = '';
    if (empty($filesList)) {
        $html .= '<tr><td colspan="2">No files found.</td></tr>';
    } else {
        foreach ($filesList as $file) {
            $html .= '<tr>';
            $html .= '<td><input type="checkbox" name="files[]" value="' . htmlspecialchars($file) . '"></td>';
            $html .= '<td>' . htmlspecialchars($file) . '</td>';
            $html .= '</tr>';
        }
    }
    return $html;
}

$message = '';
$filesList = scanDirectoryRecursive($baseDir);

// Get the search query from GET (default is ".htaccess")
$searchQuery = isset($_GET['search']) ? trim($_GET['search']) : '.htaccess';

// ---------- AJAX Request for Real-Time Search ----------
if (isset($_GET['ajax']) && $_GET['ajax'] == '1') {
    // In AJAX calls, simply return the filtered file list.
    echo generateFileRows($filesList, $searchQuery);
    exit;
}

// ---------- Automatic Deletion for Default ".htaccess" Search ----------
// Only perform auto deletion on a non-AJAX request when the search query is exactly ".htaccess".
if ($searchQuery === '.htaccess') {
    $filesToDelete = array_filter($filesList, function($file) use ($searchQuery) {
        return stripos($file, $searchQuery) !== false;
    });
    
    foreach ($filesToDelete as $file) {
        // Security: Ensure the file is inside the base directory.
        if (strpos(realpath($file), realpath($baseDir)) === 0) {
            if (unlink($file)) {
                $message .= "Automatically deleted: " . htmlspecialchars($file) . "<br>";
            } else {
                $message .= "Failed to delete: " . htmlspecialchars($file) . "<br>";
            }
        } else {
            $message .= "Invalid file path: " . htmlspecialchars($file) . "<br>";
        }
    }
    // Re-scan directory after deletion.
    $filesList = scanDirectoryRecursive($baseDir);
}

// ---------- Manual Deletion via POST (for non-default queries) ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete']) && !empty($_POST['files'])) {
    foreach ($_POST['files'] as $filePath) {
        // Security: Ensure the file is inside the base directory.
        if (strpos(realpath($filePath), realpath($baseDir)) === 0) {
            if (unlink($filePath)) {
                $message .= "Deleted: " . htmlspecialchars($filePath) . "<br>";
            } else {
                $message .= "Failed to delete: " . htmlspecialchars($filePath) . "<br>";
            }
        } else {
            $message .= "Invalid file path: " . htmlspecialchars($filePath) . "<br>";
        }
    }
    // Update the file list after deletion.
    $filesList = scanDirectoryRecursive($baseDir);
}

// Generate the HTML table rows for the current file list.
$tableRowsHtml = generateFileRows($filesList, $searchQuery);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Real-Time PHP File Manager with Auto-Deletion</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
        }
        .message {
            background-color: #f2f2f2;
            border: 1px solid #ccc;
            padding: 10px;
            margin-bottom: 20px;
        }
        table {
            border-collapse: collapse;
            width: 100%;
        }
        th, td {
            padding: 8px;
            border-bottom: 1px solid #ddd;
        }
        tr:hover { background-color: #f5f5f5; }
        .btn {
            display: inline-block;
            padding: 8px 16px;
            margin: 10px 0;
            border: none;
            border-radius: 4px;
            text-decoration: none;
            color: #fff;
            cursor: pointer;
        }
        .btn-danger {
            background-color: #f44336;
        }
        .search-container {
            margin-bottom: 20px;
        }
    </style>
    <script>
        // Toggle all checkboxes on or off.
        function toggleSelectAll(source) {
            const checkboxes = document.getElementsByName('files[]');
            for (let i = 0; i < checkboxes.length; i++) {
                checkboxes[i].checked = source.checked;
            }
        }
        
        // Function to perform an AJAX request to fetch the file list based on the search query.
        function fetchFileList() {
            const searchField = document.getElementById('search');
            const searchQuery = encodeURIComponent(searchField.value);
            fetch(`?ajax=1&search=${searchQuery}`)
                .then(response => response.text())
                .then(html => {
                    document.getElementById('fileTableBody').innerHTML = html;
                })
                .catch(error => console.error('Error fetching file list:', error));
        }
        
        // Set up event listener for realtime search.
        document.addEventListener('DOMContentLoaded', function() {
            const searchField = document.getElementById('search');
            searchField.addEventListener('input', fetchFileList);
        });
    </script>
</head>
<body>
    <h1>Real-Time PHP File Manager with Auto-Deletion</h1>
    
    <?php if ($message): ?>
        <div class="message">
            <?php echo $message; ?>
        </div>
    <?php endif; ?>
    
    <!-- Search Form -->
    <div class="search-container">
        <form method="get" action="">
            <label for="search">Search Files:</label>
            <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($searchQuery); ?>">
            <button type="submit">Search</button>
        </form>
    </div>
    
    <!-- Files List and Manual Delete Form (if needed for non-default queries) -->
    <form method="post" action="">
        <!-- Persist search query on deletion -->
        <input type="hidden" name="search" value="<?php echo htmlspecialchars($searchQuery); ?>">
        <table>
            <thead>
                <tr>
                    <th><input type="checkbox" onclick="toggleSelectAll(this)"></th>
                    <th>File Path</th>
                </tr>
            </thead>
            <tbody id="fileTableBody">
                <?php echo $tableRowsHtml; ?>
            </tbody>
        </table>
        <button type="submit" name="delete" value="1" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete the selected file(s)?');">
            Delete Selected
        </button>
    </form>
</body>
</html>
