<?php
/**
 * diagnose.php
 *
 * This script captures PHP errors during its execution and then displays a diagnostic report in HTML.
 * It also checks for a custom error log file (if one is configured) and outputs its contents.
 *
 * Usage:
 *   Place this file in your project directory.
 *   Access it via your browser to view the error report.
 */

// Turn on error reporting and ensure errors are displayed.
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Optionally, set a custom error log file (if you want to log errors to a file)
// ini_set('error_log', __DIR__ . '/error_log.txt');

// Create an array to capture errors.
$capturedErrors = [];

/**
 * Custom error handler that saves error details to the $capturedErrors array.
 *
 * @param int    $errno   The level of the error raised.
 * @param string $errstr  The error message.
 * @param string $errfile The filename that the error was raised in.
 * @param int    $errline The line number the error was raised at.
 * @return bool  Return false to allow PHP’s default error handler to also handle the error.
 */
function captureError($errno, $errstr, $errfile, $errline) {
    global $capturedErrors;
    $capturedErrors[] = [
        'date'    => date('Y-m-d H:i:s'),
        'number'  => $errno,
        'message' => $errstr,
        'file'    => $errfile,
        'line'    => $errline,
    ];
    // Return false to pass the error to the default error handler as well.
    return false;
}

// Set the custom error handler.
set_error_handler('captureError');

// ---------------------------------------------------------
// Place your code here that might generate errors...
// ---------------------------------------------------------
// For demonstration, you can uncomment the following line to trigger an error.
// trigger_error("This is a test warning.", E_USER_WARNING);

// If you want errors logged to a file, ensure that PHP is allowed to write to that location.
// Here, we check for a custom error log file that might be set in the same directory.
$customErrorLogPath = __DIR__ . '/error_log.txt';
$customErrorLogContents = file_exists($customErrorLogPath)
    ? file_get_contents($customErrorLogPath)
    : null;

// Optionally, restore the previous error handler if you want to stop capturing after this point.
// restore_error_handler();

// Begin HTML output
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>PHP Diagnostic Report</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background-color: #f9f9f9;
            color: #333;
        }
        h1, h2 {
            color: #444;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
        }
        .error-container {
            background-color: #fff;
            border: 1px solid #ddd;
            padding: 15px;
            margin-bottom: 20px;
            box-shadow: 2px 2px 6px rgba(0,0,0,0.1);
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        th, td {
            border: 1px solid #ccc;
            padding: 8px;
            text-align: left;
            font-size: 0.9em;
        }
        th {
            background-color: #eee;
        }
        pre {
            background: #f4f4f4;
            border: 1px solid #ddd;
            padding: 10px;
            overflow: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
    </style>
</head>
<body>
<div class="container">
    <h1>PHP Diagnostic Report</h1>
    
    <?php if (!empty($capturedErrors)): ?>
    <div class="error-container">
        <h2>Captured PHP Errors</h2>
        <table>
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Error Number</th>
                    <th>Error Message</th>
                    <th>File</th>
                    <th>Line</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($capturedErrors as $error): ?>
                <tr>
                    <td><?php echo htmlspecialchars($error['date']); ?></td>
                    <td><?php echo htmlspecialchars($error['number']); ?></td>
                    <td><?php echo htmlspecialchars($error['message']); ?></td>
                    <td><?php echo htmlspecialchars($error['file']); ?></td>
                    <td><?php echo htmlspecialchars($error['line']); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php else: ?>
    <div class="error-container">
        <h2>Captured PHP Errors</h2>
        <p>No PHP errors have been captured during this session.</p>
    </div>
    <?php endif; ?>

    <?php if ($customErrorLogContents !== null): ?>
    <div class="error-container">
        <h2>Custom Error Log File: <?php echo htmlspecialchars($customErrorLogPath); ?></h2>
        <pre><?php echo htmlspecialchars($customErrorLogContents); ?></pre>
    </div>
    <?php else: ?>
    <div class="error-container">
        <h2>Custom Error Log File</h2>
        <p>No custom error log file was found at <strong><?php echo htmlspecialchars($customErrorLogPath); ?></strong>.</p>
    </div>
    <?php endif; ?>
    
</div>
</body>
</html>
